/*
Copyright © 2018, github.com/satr
            2020, FR
License: MIT

ESP32 with CO2 sensor MH-Z19B (v2: 0-5000 ppm) and the display SSD1306
Data are read from the sensor's serial port.
Graph displays a change of the CO2 level on time.

  ESP32        | CO2 sensor MH-Z19B
  pin 16 (RX2) - TX
  pin 17 (TX2) - RX
  GND          - GND
  +5V          - Vin

  ESP32  | Display SSD1306
  pin 15 - Serial clock out: CLK (SCLK)
  pin  4 - Serial data out: DIN
  5V    - VCC
  GND   - GND

  ESP32  | WS2812 LED
  pin  2 - Serial data out: DIN
  5V    - VCC
  GND   - GND
   

  Notes for MH-Z19B:
  - When placed in small space, the space should be well ventilated, especially for diffusion window.
  - To ensure the normal work, power supply must be among 4.5V~5.5V DC range by not less than 150mA. 
  Out of this range, it will result in the failure sensor. 
  (The concentration output is low, or the sensor can not work normally)
*/

#include <HardwareSerial.h>
#include <Adafruit_NeoPixel.h>

#include <Arduino.h>
#include <U8g2lib.h>

#include <lmic.h>
#include <hal/hal.h>
#include <SPI.h>

#ifdef U8X8_HAVE_HW_SPI
#include <SPI.h>
#endif
#ifdef U8X8_HAVE_HW_I2C
#include <Wire.h>
#endif

#define PIXEL_PIN 2
#define PIXEL_COUNT 1

// LoRaWAN NwkSKey, network session key
// This is the default Semtech key, which is used by the early prototype TTN
// network.
static const PROGMEM u1_t NWKSKEY[16] = { 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.. };

// LoRaWAN AppSKey, application session key
// This is the default Semtech key, which is used by the early prototype TTN
// network.
static const u1_t PROGMEM APPSKEY[16] = { 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.., 0x.. };

// LoRaWAN end-device address (DevAddr)
static const u4_t DEVADDR = 0xaabbccdd ; // <-- Change this address for every node!

// These callbacks are only used in over-the-air activation, so they are
// left empty here (we cannot leave them out completely unless
// DISABLE_JOIN is set in config.h, otherwise the linker will complain).
void os_getArtEui (u1_t* buf) { }
void os_getDevEui (u1_t* buf) { }
void os_getDevKey (u1_t* buf) { }

static uint8_t loradata[] = "00";
static osjob_t sendjob;

// Schedule TX every this many seconds (might become longer due to duty
// cycle limitations).
const unsigned TX_INTERVAL = 60;

// Pin mapping
const lmic_pinmap lmic_pins = {
    .nss = 18,
    .rxtx = LMIC_UNUSED_PIN,
    .rst = 14,
    .dio = {26, 33, 32},
};

Adafruit_NeoPixel strip = Adafruit_NeoPixel(PIXEL_COUNT, PIXEL_PIN, NEO_GRB + NEO_KHZ800);

//pins description are above
// U8G2_PCD8544_84X48_F_4W_SW_SPI display(U8G2_R0, /* clock=*/ 14, /* data=*/ 13, /* cs=*/ 15, /* dc=*/ 27, /* reset=*/ 26);  // Nokia 5110 Display
U8G2_SSD1306_128X64_NONAME_F_HW_I2C display(U8G2_R0, /* reset=*/ U8X8_PIN_NONE, /* clock=*/ 15, /* data=*/ 4);   // ESP32 Thing, HW I2C with pin remapping
//put another display from this file: https://github.com/olikraus/u8g2/blob/master/tools/inoupdate/frame_buffer.ino
const byte DISPLAY_WIDTH = 128;
const byte DISPLAY_HEIGHT = 64;

HardwareSerial co2SensorSerial(1); // Serial #1

// CO2 sensor data structures:
byte cmd[9] = {0xFF,0x01,0x86,0x00,0x00,0x00,0x00,0x00,0x79}; 
/* --- Request (9 bytes) --- 
 *  0xFF - Start byte
 *  0x01 - Sensor #1
 *  0x86 - Commands: Read CO2 concentration
 *  ...
 *  0x79 - "Checksum is expected in the response"
 *  
 * --- Response (9 bytes) for the command 0x86 "Read CO2 concentration" ---
 *  0xFF - Start byte
 *  0x86 - Command
 *  0x** - CO2 concentration value (high byte)
 *  0x** - CO2 concentration value (low byte)
 *  ...
 *  0x** - Checksum value 
 *  
 * --- Other commands ---
 *  0x87 - Calibrate Zero Point (ZERO)
 *         - During the zero point calibration procedure by manual, 
 *         the sensor must work in stable gas environment (400ppm) for over 20 minutes.
 *         Connect the HD pin to low level (0V) for over 7 seconds.
 *  0x88 - Calibrate Span Point (SPAN)
 *  0x79 - ON/OFF Auto CalibrationON/OFF
 *  0x99 - Detection range setting
 *  
 */

//Response buffer
unsigned char co2SensorResponse[9]; 
const byte MH_Z19_RESPOND_START_BYTE = 0xFF;
const byte MH_Z19_RESPOND_COMMAND_READ_CO2 = 0x86;

//CO2 concentration levels
const unsigned int MIN_VALUE = 300;
const unsigned int VERY_GOOD_VALUE = 650;
const unsigned int GOOD_VALUE = 950;
const unsigned int ACCEPTABLE_VALUE = 1250;
const unsigned int MAX_VALUE = 2500;

//Display settings
unsigned int goodValueGraphValue = 0;
unsigned int acceptableValueGraphValue = 0;
const byte GRAPH_WIDTH = DISPLAY_WIDTH;
const byte GRAPH_HEIGHT = DISPLAY_HEIGHT - 20;
const unsigned int MAX_GRAPH_VALUE = 1500;
const unsigned int GRAPH_STEP = (MAX_GRAPH_VALUE - MIN_VALUE) / GRAPH_HEIGHT;
unsigned int graphData[GRAPH_WIDTH];
#define BACKCOLOR 0 // White
#define PIXELCOLOR 1  // Black
int displayHeight = DISPLAY_HEIGHT;//default - should be set in the method "initDisplay()"
byte graphSpeed = 1;
int graphDrawingCounter = 0;

//Messages
String messages[] = {"", "Sehr gut","Gut", "noch OK","Schlecht","Riskant", 
                     "Sensorfehler", "CO2 Linit", 
                     "Init ..."};
const byte MSG_EMPTY = 1;
const byte MSG_VERY_GOOD = 1;
const byte MSG_GOOD = 2;
const byte MSG_ACCEPTABLE = 3;
const byte MSG_BAD = 4;
const byte MSG_HEALTH_RISK = 5;
const byte MSG_SENSOR_ERROR = 6;
const byte MSG_NO_VALID_CO2_DATA = 7;
const byte MSG_INITIALIZING = 8;

//Current state
byte currentMessageId = 0;
unsigned int currentCo2Value = 0;


void setup() {
  currentCo2Value = 0;
  currentMessageId = MSG_INITIALIZING;

  Serial.begin(115200); //Communication between ESP32 and PC
  while(!Serial){}
  Serial.println("Serial interface initialized.");

  Serial.print("Init display...");
  initDisplay();
  Serial.println(" complete.");
  invalidateDisplay();
  initGraphStructures();
  
  Serial.print("Init serial interface to MH-Z19...");
  delay(5000);//Give some time to the sensor for starting-up
  co2SensorSerial.begin(9600, SERIAL_8N1, 16, 17); //RX, TX - communication between ESP32 and MH-Z19

  strip.begin();
  strip.setPixelColor(0, 0x10);
  strip.show();

  loradata[0] = 1;    // initial value 300
  loradata[1] = 44;

    // LMIC init
    os_init();
    // Reset the MAC state. Session and pending data transfers will be discarded.
    LMIC_reset();

    // Set static session parameters. Instead of dynamically establishing a session
    // by joining the network, precomputed session parameters are be provided.
    #ifdef PROGMEM
    // On AVR, these values are stored in flash and only copied to RAM
    // once. Copy them to a temporary buffer here, LMIC_setSession will
    // copy them into a buffer of its own again.
    uint8_t appskey[sizeof(APPSKEY)];
    uint8_t nwkskey[sizeof(NWKSKEY)];
    memcpy_P(appskey, APPSKEY, sizeof(APPSKEY));
    memcpy_P(nwkskey, NWKSKEY, sizeof(NWKSKEY));
    LMIC_setSession (0x1, DEVADDR, nwkskey, appskey);
    #else
    // If not running an AVR with PROGMEM, just use the arrays directly
    LMIC_setSession (0x1, DEVADDR, NWKSKEY, APPSKEY);
    #endif

    #if defined(CFG_eu868)
    // Set up the channels used by the Things Network, which corresponds
    // to the defaults of most gateways. Without this, only three base
    // channels from the LoRaWAN specification are used, which certainly
    // works, so it is good for debugging, but can overload those
    // frequencies, so be sure to configure the full frequency range of
    // your network here (unless your network autoconfigures them).
    // Setting up channels should happen after LMIC_setSession, as that
    // configures the minimal channel set.
    // NA-US channels 0-71 are configured automatically
    LMIC_setupChannel(0, 868100000, DR_RANGE_MAP(DR_SF12, DR_SF7),  BAND_CENTI);      // g-band
    LMIC_setupChannel(1, 868300000, DR_RANGE_MAP(DR_SF12, DR_SF7B), BAND_CENTI);      // g-band
    LMIC_setupChannel(2, 868500000, DR_RANGE_MAP(DR_SF12, DR_SF7),  BAND_CENTI);      // g-band
    LMIC_setupChannel(3, 867100000, DR_RANGE_MAP(DR_SF12, DR_SF7),  BAND_CENTI);      // g-band
    LMIC_setupChannel(4, 867300000, DR_RANGE_MAP(DR_SF12, DR_SF7),  BAND_CENTI);      // g-band
    LMIC_setupChannel(5, 867500000, DR_RANGE_MAP(DR_SF12, DR_SF7),  BAND_CENTI);      // g-band
    LMIC_setupChannel(6, 867700000, DR_RANGE_MAP(DR_SF12, DR_SF7),  BAND_CENTI);      // g-band
    LMIC_setupChannel(7, 867900000, DR_RANGE_MAP(DR_SF12, DR_SF7),  BAND_CENTI);      // g-band
    LMIC_setupChannel(8, 868800000, DR_RANGE_MAP(DR_FSK,  DR_FSK),  BAND_MILLI);      // g2-band
    // TTN defines an additional channel at 869.525Mhz using SF9 for class B
    // devices' ping slots. LMIC does not have an easy way to define set this
    // frequency and support for class B is spotty and untested, so this
    // frequency is not configured here.
    #elif defined(CFG_us915)
    // NA-US channels 0-71 are configured automatically
    // but only one group of 8 should (a subband) should be active
    // TTN recommends the second sub band, 1 in a zero based count.
    // https://github.com/TheThingsNetwork/gateway-conf/blob/master/US-global_conf.json
    LMIC_selectSubBand(1);
    #endif

    // Disable link check validation
    LMIC_setLinkCheckMode(0);

    // TTN uses SF9 for its RX2 window.
    LMIC.dn2Dr = DR_SF9;

    // Set data rate and transmit power for uplink (note: txpow seems to be ignored by the library)
    LMIC_setDrTxpow(DR_SF7,14);

    // Start job
    do_send(&sendjob);

  
  Serial.println(" complete");
  Serial.println("Initialization complete.");
}


void loop() 
{
  readCo2SensorValueToCo2Response();
  
  if(validateCo2Response()) {
    setCo2ValueAndCurrentMessage((256 * (unsigned int) co2SensorResponse[2]) + (unsigned int) co2SensorResponse[3]);
  } else {
    currentMessageId = MSG_SENSOR_ERROR;
  }
  
  invalidateDisplay();
  
  printRespond();

  os_runloop_once();
  
  delay(2000);//timeout between reads (in milliseconds)
}


void onEvent (ev_t ev) {
    Serial.print(os_getTime());
    Serial.print(": ");
    switch(ev) {
        case EV_SCAN_TIMEOUT:
            Serial.println(F("EV_SCAN_TIMEOUT"));
            break;
        case EV_BEACON_FOUND:
            Serial.println(F("EV_BEACON_FOUND"));
            break;
        case EV_BEACON_MISSED:
            Serial.println(F("EV_BEACON_MISSED"));
            break;
        case EV_BEACON_TRACKED:
            Serial.println(F("EV_BEACON_TRACKED"));
            break;
        case EV_JOINING:
            Serial.println(F("EV_JOINING"));
            break;
        case EV_JOINED:
            Serial.println(F("EV_JOINED"));
            break;
        case EV_RFU1:
            Serial.println(F("EV_RFU1"));
            break;
        case EV_JOIN_FAILED:
            Serial.println(F("EV_JOIN_FAILED"));
            break;
        case EV_REJOIN_FAILED:
            Serial.println(F("EV_REJOIN_FAILED"));
            break;
        case EV_TXCOMPLETE:
            Serial.println(F("EV_TXCOMPLETE (includes waiting for RX windows)"));
            if (LMIC.txrxFlags & TXRX_ACK)
              Serial.println(F("Received ack"));
            if (LMIC.dataLen) {
              Serial.println(F("Received "));
              Serial.println(LMIC.dataLen);
              Serial.println(F(" bytes of payload"));
            }
            // Schedule next transmission
            os_setTimedCallback(&sendjob, os_getTime()+sec2osticks(TX_INTERVAL), do_send);
            break;
        case EV_LOST_TSYNC:
            Serial.println(F("EV_LOST_TSYNC"));
            break;
        case EV_RESET:
            Serial.println(F("EV_RESET"));
            break;
        case EV_RXCOMPLETE:
            // data received in ping slot
            Serial.println(F("EV_RXCOMPLETE"));
            break;
        case EV_LINK_DEAD:
            Serial.println(F("EV_LINK_DEAD"));
            break;
        case EV_LINK_ALIVE:
            Serial.println(F("EV_LINK_ALIVE"));
            break;
         default:
            Serial.println(F("Unknown event"));
            break;
    }
}

void do_send(osjob_t* j){
    // Check if there is not a current TX/RX job running
    if (LMIC.opmode & OP_TXRXPEND) {
        Serial.println(F("OP_TXRXPEND, not sending"));
    } else {
        // Prepare upstream data transmission at the next possible time.
        LMIC_setTxData2(1, loradata, sizeof(loradata)-1, 0);
        Serial.println(F("Packet queued"));
    }
    // Next TX is scheduled after TX_COMPLETE event.
}




uint32_t Wheel(byte Wert, byte Rdimm, byte Gdimm, byte Bdimm)
{
  uint16_t Phase, Blue, Green, Red, Level;
  if(Wert > 250) Wert = 250;
  Phase = Wert / 42;
  Level = Wert % 42;
 
  switch ( Phase )
  {
    case 0:                 //
      {
        Red   = 255;
        Green = Level * 6;
        Blue  = 0;
      }
      break;
 
    case 1:
      {
        Red   = 255 - Level * 6;
        Green = 255;
        Blue  = 0;
      }
      break;
 
    case 2:
      {
        Red   = 0;
        Green = 255;
        Blue  = Level * 6;
      }
      break;
 
    case 3:
      {
        Red   = 0;
        Green = 255 - Level * 6;
        Blue  = 255;
      }
      break;
 
    case 4:
      {
        Red   = Level * 6;
        Green = 0;
        Blue  = 255;
      }
      break;
    case 5:
      {
        Red   = 255;
        Green = 0;
        Blue  = 255 - Level * 6;
      }
      break;
 
    default:
      Red = Green = Blue = 255;
  }
  return strip.Color(Red * Rdimm / 100, Green * Gdimm / 100, Blue * Bdimm / 100);
}

void initGraphStructures() {
  for(byte i = 0; i < GRAPH_WIDTH; i++) {
    graphData[i] = getGraphValue(MIN_VALUE);
  }
  goodValueGraphValue = getGraphValue(GOOD_VALUE);
  acceptableValueGraphValue = getGraphValue(ACCEPTABLE_VALUE);
}

int getGraphValue(unsigned int value) {
    if(value < MIN_VALUE) {
      return 0;
    }
    if(value > MAX_GRAPH_VALUE) {
      value = MAX_GRAPH_VALUE;
    }
    return (value - MIN_VALUE) / GRAPH_STEP;
}

bool validateCo2Response() {
  byte checksum = 0;
  for (byte i = 1; i < 8; i++) {
    checksum += co2SensorResponse[i];
  }
  checksum = 0xFF - checksum;
  checksum++;
  bool valid = co2SensorResponse[0] == MH_Z19_RESPOND_START_BYTE 
               && co2SensorResponse[1] == MH_Z19_RESPOND_COMMAND_READ_CO2
               && co2SensorResponse[8] == checksum;
  if(!valid) {
    Serial.println("CRC error: " + String(checksum) + "/"+ String(co2SensorResponse[8]));
  }
  return valid; 
}
  
void readCo2SensorValueToCo2Response() {
  co2SensorSerial.write(cmd, 9);
  memset(co2SensorResponse, 0, 9);
  if(co2SensorSerial.available()) {
    co2SensorSerial.readBytes(co2SensorResponse, 9);
  }
}

void printRespond() {
  Serial.print("Respond: ");
  for(byte i = 0; i < 9; i++) {
    Serial.print(co2SensorResponse[i], HEX);
    Serial.print(" ");
  }
  Serial.println("");
}

void addValueToGraph(unsigned int value) {
  for(int i = GRAPH_WIDTH - 1; i > 0; i--) {
    graphData[i] = graphData[i - 1];//shift graph values
  }
  graphData[0] = getGraphValue(value);
}
  
void setCo2ValueAndCurrentMessage(unsigned int co2Value) {
  long tempco2;
  
  currentCo2Value = co2Value;
  addValueToGraph(co2Value);
  
  if (co2Value <= MIN_VALUE || co2Value > 4900) {
    if (co2Value > 4900) currentCo2Value = 4900;
    else currentCo2Value = 0;
    currentMessageId = MSG_NO_VALID_CO2_DATA;
  } else {
    currentCo2Value = co2Value; 
    currentMessageId = getMessageByCo2Value(co2Value);

    // Ampel-LED-Farbe entsprechend CO2-Wert einstellen 
    tempco2 = 150 - (150 * currentCo2Value / 1500);
    if (tempco2 < -40) tempco2 = -40;
    strip.setPixelColor(0,Wheel(tempco2,25,25,25));
    strip.show();

    loradata[0] = (int)currentCo2Value >> 8;
    loradata[1] = (int)currentCo2Value & 0xFF;
    
  }

  Serial.println("CO2:" + String(co2Value) + "; \"" + messages[currentMessageId] + "\"");
}

byte getMessageByCo2Value(unsigned int co2Value){
  if (co2Value < VERY_GOOD_VALUE) {   
    return MSG_VERY_GOOD;
  }
  if (co2Value < GOOD_VALUE) {   
    return MSG_GOOD;
  }
  if (co2Value < ACCEPTABLE_VALUE) {   
    return MSG_ACCEPTABLE;
  }
  if (co2Value < MAX_VALUE) {   
    return MSG_BAD;
  }
  return MSG_HEALTH_RISK;
}

void setFont() {  
  display.setFont(u8g2_font_6x10_tf);
  display.setFontRefHeightExtendedText();
  display.setDrawColor(PIXELCOLOR);
  display.setFontPosTop();
  display.setFontDirection(0);
}

void drawCo2Value() {
  String co2StatusText = "CO2 " + String(currentCo2Value) + " ppm";
  char co2ValueBuff[sizeof(co2StatusText) + 1];
  String(co2StatusText).toCharArray(co2ValueBuff, sizeof(co2ValueBuff));
  display.drawStr(0, 0, co2ValueBuff);
}

void drawMessage() {
  char msgBuff[sizeof(messages[currentMessageId]) + 1];
  messages[currentMessageId].toCharArray(msgBuff, sizeof(msgBuff));
  display.drawStr(0, 10, msgBuff);
}

void drawGraph() {
  bool dashState = false;
  for(int i = 0; i < GRAPH_WIDTH; i++) {
    byte x = GRAPH_WIDTH - 1 - i;
    display.setDrawColor(PIXELCOLOR);
    display.drawLine(x, displayHeight - 1, x, displayHeight - 1 - graphData[i]);
    display.setDrawColor(dashState ? PIXELCOLOR : BACKCOLOR);
    display.drawPixel(x, displayHeight - 1 - goodValueGraphValue);
    display.drawPixel(x, displayHeight - 1 - acceptableValueGraphValue);
    dashState = !dashState;
  }
}

void draw() {
  setFont();
  drawCo2Value();  
  drawMessage();
  drawGraph();
}

void invalidateDisplay() {
  display.clearBuffer();
  draw();
  display.sendBuffer();
}

void initDisplay() {
  display.begin();
  display.setPowerSave(0); ///Not sure yet
  displayHeight = DISPLAY_HEIGHT;// quick-fix - it shoul be set from display.height();
}
